// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package samreg

import (
	"errors"
	"slices"
	"strings"
	"testing"
)

func TestUsername(t *testing.T) {
	tests := []struct {
		name        string
		userVBuffer []byte
		userF       *userF
		want        string
		wantErr     bool
		wantErrText string
	}{
		{
			name:        "returns_username",
			userVBuffer: []byte(strings.Repeat("\x00", 16) + "\x08" + strings.Repeat("\x00", 187) + "\x42\x00\x43\x00\x44\x00\x45\x00"),
			want:        "BCDE",
		},
		{
			name:        "userV_parse_failure_returns_error",
			userVBuffer: []byte(strings.Repeat("\xFF", 0xCC)),
			wantErr:     true,
			wantErrText: errReadOutOfBounds.Error(),
		},
	}

	for _, tc := range tests {
		t.Run(tc.name, func(t *testing.T) {
			userV, err := newUserV(tc.userVBuffer, "irrelevant")
			if err != nil {
				t.Fatalf("Failed to create userV for tests: %v", err)
			}

			userinfo := UserInfo{
				userV: userV,
				userF: tc.userF,
			}

			got, gotErr := userinfo.Username()
			if (gotErr != nil) != tc.wantErr {
				t.Errorf("Username(): unexpected error: %v", gotErr)
			}

			if tc.wantErr {
				if !strings.Contains(gotErr.Error(), tc.wantErrText) {
					t.Errorf("Username(): unexpected error, got: %v, want: %v", gotErr, tc.wantErr)
				}

				return
			}

			if got != tc.want {
				t.Errorf("Username(): got %v, want %v", got, tc.want)
			}
		})
	}
}

func TestEnabled(t *testing.T) {
	tests := []struct {
		name    string
		userV   *userV
		userF   *userF
		want    bool
		wantErr error
	}{
		{
			name:  "user_enabled_returns_true",
			userV: nil,
			userF: newUserF([]byte(strings.Repeat("A", 56)+"\x00"), "irrelevant"),
			want:  true,
		},
		{
			name:  "user_disabled_returns_false",
			userV: nil,
			userF: newUserF([]byte(strings.Repeat("A", 56)+"\x01"), "irrelevant"),
			want:  false,
		},
		{
			name:    "userF_parsing_failure_returns_error",
			userV:   nil,
			userF:   newUserF([]byte(""), "irrelevant"),
			wantErr: errUserFTooShort,
		},
	}

	for _, tc := range tests {
		t.Run(tc.name, func(t *testing.T) {
			userinfo := UserInfo{
				userV: tc.userV,
				userF: tc.userF,
			}

			got, gotErr := userinfo.Enabled()
			if !errors.Is(gotErr, tc.wantErr) {
				t.Errorf("Enabled(): unexpected error, got: %v, want: %v", gotErr, tc.wantErr)
			}

			if tc.wantErr != nil {
				return
			}

			if got != tc.want {
				t.Errorf("Enabled(): got %v, want %v", got, tc.want)
			}
		})
	}
}

func TestHashes(t *testing.T) {
	tests := []struct {
		name        string
		userVBuffer []byte
		rid         string
		syskey      []byte
		wantLM      []byte
		wantNT      []byte
		wantErr     bool
		wantErrText string
	}{
		{
			name:        "hash_with_RC4_succeeds",
			rid:         "000003E9",
			userVBuffer: []byte("\x00\x00\x00\x00\xd4\x00\x00\x00\x02\x00\x01\x00\xd4\x00\x00\x00\x0c\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0\x00\x00\x00\x08\x00\x00\x00\x01\x00\x00\x00\xe8\x00\x00\x00\x14\x00\x00\x00\x00\x00\x00\x00\xfc\x00\x00\x00\x14\x00\x00\x00\x00\x00\x00\x00\x10\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x14\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x01\x00\x14\x80\xb4\x00\x00\x00\xc4\x00\x00\x00\x14\x00\x00\x00\x44\x00\x00\x00\x02\x00\x30\x00\x02\x00\x00\x00\x02\xc0\x14\x00\x44\x00\x05\x01\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x02\xc0\x14\x00\xff\xff\x1f\x00\x01\x01\x00\x00\x00\x00\x00\x05\x07\x00\x00\x00\x02\x00\x70\x00\x04\x00\x00\x00\x00\x00\x14\x00\x5b\x03\x02\x00\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x18\x00\xff\x07\x0f\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x00\x00\x18\x00\xff\x07\x0f\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x24\x02\x00\x00\x00\x00\x24\x00\x44\x00\x02\x00\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\xee\x9d\xd5\xc6\x43\xed\x3f\x53\x2e\xc8\x5e\x0a\xe9\x03\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x6c\x00\x6d\x00\x75\x00\x73\x00\x65\x00\x72\x00\x01\x02\x00\x00\x07\x00\x00\x00\x03\x00\x01\x00\x0c\x4b\x53\xdf\x15\xed\xf9\xeb\x41\x05\xfe\xca\x48\x02\x70\x55\x03\x00\x01\x00\x72\x10\xad\x76\x18\xf1\x1f\x38\x49\x24\x6d\x13\x38\x15\x8b\x03\x03\x00\x01\x00\x03\x00\x01\x00"),
			syskey:      []byte("\x3d\x21\x2c\xe8\xa2\xda\x83\x43\xbd\xad\x1e\xf2\xcf\xb6\xb3\x1c"),
			wantLM:      []byte("\xe5\x2c\xac\x67\x41\x9a\x9a\x22\x66\x43\x45\x14\x0a\x85\x2f\x61"),
			wantNT:      []byte("\x58\xa4\x78\x13\x5a\x93\xac\x3b\xf0\x58\xa5\xea\x0e\x8f\xdb\x71"),
		},
		{
			name:        "hash_with_RC4_only_NTLM_succeeds",
			rid:         "000001F4",
			userVBuffer: []byte("\x00\x00\x00\x00\xbc\x00\x00\x00\x02\x00\x01\x00\xbc\x00\x00\x00\x1a\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x00\x00\x6c\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x15\x00\x00\x00\xa8\x00\x00\x00\x5c\x01\x00\x00\x08\x00\x00\x00\x01\x00\x00\x00\x64\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x68\x01\x00\x00\x14\x00\x00\x00\x00\x00\x00\x00\x7c\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x80\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x01\x00\x14\x80\x9c\x00\x00\x00\xac\x00\x00\x00\x14\x00\x00\x00\x44\x00\x00\x00\x02\x00\x30\x00\x02\x00\x00\x00\x02\xc0\x14\x00\x44\x00\x05\x01\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x02\xc0\x14\x00\xff\xff\x1f\x00\x01\x01\x00\x00\x00\x00\x00\x05\x07\x00\x00\x00\x02\x00\x58\x00\x03\x00\x00\x00\x00\x00\x14\x00\x5b\x03\x02\x00\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x18\x00\xff\x07\x0f\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x00\x00\x24\x00\x44\x00\x02\x00\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\xee\x9d\xd5\xc6\x43\xed\x3f\x53\x2e\xc8\x5e\x0a\xf4\x01\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x41\x00\x64\x00\x6d\x00\x69\x00\x6e\x00\x69\x00\x73\x00\x74\x00\x72\x00\x61\x00\x74\x00\x6f\x00\x72\x00\x64\x00\x42\x00\x75\x00\x69\x00\x6c\x00\x74\x00\x2d\x00\x69\x00\x6e\x00\x20\x00\x61\x00\x63\x00\x63\x00\x6f\x00\x75\x00\x6e\x00\x74\x00\x20\x00\x66\x00\x6f\x00\x72\x00\x20\x00\x61\x00\x64\x00\x6d\x00\x69\x00\x6e\x00\x69\x00\x73\x00\x74\x00\x65\x00\x72\x00\x69\x00\x6e\x00\x67\x00\x20\x00\x74\x00\x68\x00\x65\x00\x20\x00\x63\x00\x6f\x00\x6d\x00\x70\x00\x75\x00\x74\x00\x65\x00\x72\x00\x2f\x00\x64\x00\x6f\x00\x6d\x00\x61\x00\x69\x00\x6e\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x01\x00\x01\x02\x00\x00\x07\x00\x00\x00\x03\x00\x01\x00\x03\x00\x01\x00\xed\x92\x87\x92\x78\x3b\x69\x2c\x21\x37\x49\xbc\xdb\xe3\x1a\xf5\x03\x00\x01\x00\x03\x00\x01\x00"),
			syskey:      []byte("\x3d\x21\x2c\xe8\xa2\xda\x83\x43\xbd\xad\x1e\xf2\xcf\xb6\xb3\x1c"),
			wantLM:      []byte(""),
			wantNT:      []byte("\x58\xa4\x78\x13\x5a\x93\xac\x3b\xf0\x58\xa5\xea\x0e\x8f\xdb\x71"),
		},
		{
			name:        "hash_with_AES_succeeds",
			rid:         "000003EA",
			userVBuffer: []byte("\x00\x00\x00\x00\x0c\x01\x00\x00\x03\x00\x01\x00\x0c\x01\x00\x00\x0c\x00\x00\x00\x00\x00\x00\x00\x18\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x18\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x18\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x18\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x18\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x18\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x18\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x18\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x18\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x18\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x18\x01\x00\x00\x08\x00\x00\x00\x01\x00\x00\x00\x20\x01\x00\x00\x38\x00\x00\x00\x00\x00\x00\x00\x58\x01\x00\x00\x38\x00\x00\x00\x00\x00\x00\x00\x90\x01\x00\x00\x18\x00\x00\x00\x00\x00\x00\x00\xa8\x01\x00\x00\x18\x00\x00\x00\x00\x00\x00\x00\x01\x00\x14\x80\xec\x00\x00\x00\xfc\x00\x00\x00\x14\x00\x00\x00\x44\x00\x00\x00\x02\x00\x30\x00\x02\x00\x00\x00\x02\xc0\x14\x00\x44\x00\x05\x01\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x02\xc0\x14\x00\xff\xff\x1f\x00\x01\x01\x00\x00\x00\x00\x00\x05\x07\x00\x00\x00\x02\x00\xa8\x00\x05\x00\x00\x00\x00\x00\x14\x00\x5b\x03\x02\x00\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x18\x00\xff\x07\x0f\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x00\x00\x18\x00\xff\x07\x0f\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x24\x02\x00\x00\x00\x00\x38\x00\x1b\x03\x02\x00\x01\x0a\x00\x00\x00\x00\x00\x0f\x03\x00\x00\x00\x00\x04\x00\x00\xde\xa2\x28\x67\x21\x3e\xd2\xaf\x19\xad\x5d\x79\xb0\xc1\x07\x29\x27\x56\xfc\x20\xd8\xad\x66\xf6\x10\xf2\x68\xfa\xdf\x2a\xf8\x0f\x00\x00\x24\x00\x44\x00\x02\x00\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\xed\xfd\x54\x5c\x76\xc8\xf0\x10\x1c\xda\xe3\x70\xea\x03\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x6c\x00\x6d\x00\x75\x00\x73\x00\x65\x00\x72\x00\x01\x02\x00\x00\x07\x00\x00\x00\x02\x00\x02\x00\x10\x00\x00\x00\x6a\xea\xb0\x62\x73\x97\xf9\xd0\x0d\x5d\x50\xcc\x3d\xef\xa8\xec\xed\xa9\x97\x59\x01\xe9\xa0\x17\x91\x9f\x8d\xf2\x52\x01\xd9\x6f\x69\x0f\x88\xe0\x2c\xe9\x22\x81\x9b\xf4\x14\x8f\x4b\xf9\x26\x8c\x02\x00\x02\x00\x10\x00\x00\x00\x86\xc1\x0a\xea\xc7\x24\x06\xdb\xb2\x9f\x09\x42\x87\xcf\xab\xbd\xed\xb0\x32\x75\xfb\x5d\xaf\x0a\xff\x48\xe3\x91\x51\x28\xff\xaa\x20\x93\xb3\x55\x53\xf1\x7d\x23\xde\xe0\xa6\xac\xaa\x27\x1e\xdd\x02\x00\x02\x00\x00\x00\x00\x00\x4e\x4d\xe5\x26\xdc\x17\xfa\x6f\x26\xb1\x59\xa6\xec\x09\x94\xae\x02\x00\x02\x00\x00\x00\x00\x00\x4b\x5e\x9a\x54\xa0\xc5\x6c\x8a\x07\x4e\x31\xfa\xf0\x62\xd1\xef"),
			syskey:      []byte("\xfc\xde\xe8\x3a\xc6\xc1\x4b\x28\xf5\x26\x50\x1f\xc6\xe8\xbb\xc3"),
			wantLM:      []byte("\xe5\x2c\xac\x67\x41\x9a\x9a\x22\x66\x43\x45\x14\x0a\x85\x2f\x61"),
			wantNT:      []byte("\x58\xa4\x78\x13\x5a\x93\xac\x3b\xf0\x58\xa5\xea\x0e\x8f\xdb\x71"),
		},
		{
			name:        "hash_with_AES_only_NTLM_succeeds",
			rid:         "000001F4",
			userVBuffer: []byte("\x00\x00\x00\x00\xf4\x00\x00\x00\x03\x00\x01\x00\xf4\x00\x00\x00\x1a\x00\x00\x00\x00\x00\x00\x00\x10\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x01\x00\x00\x6c\x00\x00\x00\x00\x00\x00\x00\x7c\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x7c\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x7c\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x7c\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x7c\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x7c\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x7c\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x7c\x01\x00\x00\x15\x00\x00\x00\xa8\x00\x00\x00\x94\x01\x00\x00\x08\x00\x00\x00\x01\x00\x00\x00\x9c\x01\x00\x00\x18\x00\x00\x00\x00\x00\x00\x00\xb4\x01\x00\x00\x38\x00\x00\x00\x00\x00\x00\x00\xec\x01\x00\x00\x18\x00\x00\x00\x00\x00\x00\x00\x04\x02\x00\x00\x18\x00\x00\x00\x00\x00\x00\x00\x01\x00\x14\x80\xd4\x00\x00\x00\xe4\x00\x00\x00\x14\x00\x00\x00\x44\x00\x00\x00\x02\x00\x30\x00\x02\x00\x00\x00\x02\xc0\x14\x00\x44\x00\x05\x01\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x02\xc0\x14\x00\xff\xff\x1f\x00\x01\x01\x00\x00\x00\x00\x00\x05\x07\x00\x00\x00\x02\x00\x90\x00\x04\x00\x00\x00\x00\x00\x14\x00\x5b\x03\x02\x00\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x18\x00\xff\x07\x0f\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x00\x00\x38\x00\x1b\x03\x02\x00\x01\x0a\x00\x00\x00\x00\x00\x0f\x03\x00\x00\x00\x00\x04\x00\x00\xde\xa2\x28\x67\x21\x3e\xd2\xaf\x19\xad\x5d\x79\xb0\xc1\x07\x29\x27\x56\xfc\x20\xd8\xad\x66\xf6\x10\xf2\x68\xfa\xdf\x2a\xf8\x0f\x00\x00\x24\x00\x44\x00\x02\x00\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\xed\xfd\x54\x5c\x76\xc8\xf0\x10\x1c\xda\xe3\x70\xf4\x01\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x41\x00\x64\x00\x6d\x00\x69\x00\x6e\x00\x69\x00\x73\x00\x74\x00\x72\x00\x61\x00\x74\x00\x6f\x00\x72\x00\x64\x00\x42\x00\x75\x00\x69\x00\x6c\x00\x74\x00\x2d\x00\x69\x00\x6e\x00\x20\x00\x61\x00\x63\x00\x63\x00\x6f\x00\x75\x00\x6e\x00\x74\x00\x20\x00\x66\x00\x6f\x00\x72\x00\x20\x00\x61\x00\x64\x00\x6d\x00\x69\x00\x6e\x00\x69\x00\x73\x00\x74\x00\x65\x00\x72\x00\x69\x00\x6e\x00\x67\x00\x20\x00\x74\x00\x68\x00\x65\x00\x20\x00\x63\x00\x6f\x00\x6d\x00\x70\x00\x75\x00\x74\x00\x65\x00\x72\x00\x2f\x00\x64\x00\x6f\x00\x6d\x00\x61\x00\x69\x00\x6e\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xb2\xb9\x06\x01\x02\x00\x00\x07\x00\x00\x00\x02\x00\x02\x00\x00\x00\x00\x00\x36\xf4\xab\xc3\x1a\x2c\xf9\x76\x09\xa4\xcb\xbb\xe4\xbc\xac\x1c\x02\x00\x02\x00\x10\x00\x00\x00\xa3\x28\x48\xec\x7d\x73\x12\xec\x81\xeb\x50\xd0\x65\x09\x55\xd4\x48\xf2\xb6\x8b\xd9\x06\xa2\xbd\xb2\xaf\x39\x1c\xe2\x60\x44\x56\x6b\x80\x62\xb6\x55\xf4\x2b\x05\x9d\xfb\x5c\x68\x55\x4a\x5b\xc3\x02\x00\x02\x00\x00\x00\x00\x00\xf4\x62\x28\xbc\x7b\xf3\x36\xe7\xaf\x0f\x7d\xf4\x88\xd6\x78\x07\x02\x00\x02\x00\x00\x00\x00\x00\x0f\x24\x35\xc2\x94\x02\xd9\x4f\x28\xe0\xe7\x92\x86\x92\xde\x0b"),
			syskey:      []byte("\xfc\xde\xe8\x3a\xc6\xc1\x4b\x28\xf5\x26\x50\x1f\xc6\xe8\xbb\xc3"),
			wantLM:      []byte(""),
			wantNT:      []byte("\x58\xa4\x78\x13\x5a\x93\xac\x3b\xf0\x58\xa5\xea\x0e\x8f\xdb\x71"),
		},
		{
			name:        "userV_parsing_failure_returns_error",
			rid:         "000001F4",
			userVBuffer: []byte(strings.Repeat("\x00", 0xCC)),
			wantErr:     true,
			wantErrText: errNoHashInfoFound.Error(),
		},
		{
			name:        "decryption_failures_return_error",
			rid:         "",
			userVBuffer: []byte("\x00\x00\x00\x00\xbc\x00\x00\x00\x02\x00\x01\x00\xbc\x00\x00\x00\x1a\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\x00\x00\x00\x6c\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x01\x00\x00\x15\x00\x00\x00\xa8\x00\x00\x00\x5c\x01\x00\x00\x08\x00\x00\x00\x01\x00\x00\x00\x64\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x68\x01\x00\x00\x14\x00\x00\x00\x00\x00\x00\x00\x7c\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x80\x01\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x01\x00\x14\x80\x9c\x00\x00\x00\xac\x00\x00\x00\x14\x00\x00\x00\x44\x00\x00\x00\x02\x00\x30\x00\x02\x00\x00\x00\x02\xc0\x14\x00\x44\x00\x05\x01\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x02\xc0\x14\x00\xff\xff\x1f\x00\x01\x01\x00\x00\x00\x00\x00\x05\x07\x00\x00\x00\x02\x00\x58\x00\x03\x00\x00\x00\x00\x00\x14\x00\x5b\x03\x02\x00\x01\x01\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x18\x00\xff\x07\x0f\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x00\x00\x24\x00\x44\x00\x02\x00\x01\x05\x00\x00\x00\x00\x00\x05\x15\x00\x00\x00\xee\x9d\xd5\xc6\x43\xed\x3f\x53\x2e\xc8\x5e\x0a\xf4\x01\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x01\x02\x00\x00\x00\x00\x00\x05\x20\x00\x00\x00\x20\x02\x00\x00\x41\x00\x64\x00\x6d\x00\x69\x00\x6e\x00\x69\x00\x73\x00\x74\x00\x72\x00\x61\x00\x74\x00\x6f\x00\x72\x00\x64\x00\x42\x00\x75\x00\x69\x00\x6c\x00\x74\x00\x2d\x00\x69\x00\x6e\x00\x20\x00\x61\x00\x63\x00\x63\x00\x6f\x00\x75\x00\x6e\x00\x74\x00\x20\x00\x66\x00\x6f\x00\x72\x00\x20\x00\x61\x00\x64\x00\x6d\x00\x69\x00\x6e\x00\x69\x00\x73\x00\x74\x00\x65\x00\x72\x00\x69\x00\x6e\x00\x67\x00\x20\x00\x74\x00\x68\x00\x65\x00\x20\x00\x63\x00\x6f\x00\x6d\x00\x70\x00\x75\x00\x74\x00\x65\x00\x72\x00\x2f\x00\x64\x00\x6f\x00\x6d\x00\x61\x00\x69\x00\x6e\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x01\x00\x01\x02\x00\x00\x07\x00\x00\x00\x03\x00\x01\x00\x03\x00\x01\x00\xed\x92\x87\x92\x78\x3b\x69\x2c\x21\x37\x49\xbc\xdb\xe3\x1a\xf5\x03\x00\x01\x00\x03\x00\x01\x00"),
			wantErr:     true,
			wantErrText: errInvalidRIDSize.Error(),
		},
	}

	for _, tc := range tests {
		t.Run(tc.name, func(t *testing.T) {
			userV, err := newUserV(tc.userVBuffer, tc.rid)
			if err != nil {
				t.Fatalf("Failed to create userV for tests: %v", err)
			}

			userinfo := UserInfo{
				rid:   tc.rid,
				userV: userV,
			}

			gotLM, gotNT, gotErr := userinfo.Hashes(tc.syskey)
			if (gotErr != nil) != tc.wantErr {
				t.Errorf("Hashes(): unexpected error: %v", gotErr)
			}

			if tc.wantErr {
				if !strings.Contains(gotErr.Error(), tc.wantErrText) {
					t.Errorf("Hashes(): unexpected error, got: %v, want: %v", gotErr, tc.wantErr)
				}

				return
			}

			if !slices.Equal(gotLM, tc.wantLM) {
				t.Errorf("Hashes(): got LM hash %x, want %x", gotLM, tc.wantLM)
			}

			if !slices.Equal(gotNT, tc.wantNT) {
				t.Errorf("Hashes(): got NT hash %x, want %x", gotNT, tc.wantNT)
			}
		})
	}
}
