// Copyright (c) HashiCorp, Inc.
// SPDX-License-Identifier: MPL-2.0

package logging

import (
	"context"

	"github.com/hashicorp/go-hclog"
)

// GetSDKRootLogger returns the root logger used for writing logs from an SDK.
// If no root logger has been created, it will return nil.
func GetSDKRootLogger(ctx context.Context) hclog.Logger {
	logger := ctx.Value(SDKRootLoggerKey)
	if logger == nil {
		return nil
	}

	hclogger, ok := logger.(hclog.Logger)
	if !ok {
		return nil
	}

	return hclogger
}

// GetSDKRootLoggerOptions returns the root logger options used for
// creating the root SDK logger. If the root logger has not been created or
// the options are not present, it will return nil.
func GetSDKRootLoggerOptions(ctx context.Context) *hclog.LoggerOptions {
	if GetSDKRootLogger(ctx) == nil {
		return nil
	}

	loggerOptionsRaw := ctx.Value(SDKRootLoggerOptionsKey)

	if loggerOptionsRaw == nil {
		return nil
	}

	loggerOptions, ok := loggerOptionsRaw.(*hclog.LoggerOptions)

	if !ok {
		return nil
	}

	return loggerOptions
}

// SetSDKRootLogger sets `logger` as the root logger used for writing logs from
// an SDK.
func SetSDKRootLogger(ctx context.Context, logger hclog.Logger) context.Context {
	return context.WithValue(ctx, SDKRootLoggerKey, logger)
}

// SetSDKRootLoggerOptions sets `loggerOptions` as the root logger options
// used for creating the SDK root logger.
func SetSDKRootLoggerOptions(ctx context.Context, loggerOptions *hclog.LoggerOptions) context.Context {
	return context.WithValue(ctx, SDKRootLoggerOptionsKey, loggerOptions)
}

// GetSDKRootTFLoggerOpts retrieves the LoggerOpts of the SDK root logger.
// The value is stored in the context.Context: if none is found, a new one will be created.
func GetSDKRootTFLoggerOpts(ctx context.Context) LoggerOpts {
	lOpts, ok := ctx.Value(sdkRootTFLoggerOptsKey()).(LoggerOpts)
	if !ok {
		lOpts = LoggerOpts{}
	}

	return lOpts
}

// SetSDKRootTFLoggerOpts sets the LoggerOpts of the SDK root logger, in the context.
func SetSDKRootTFLoggerOpts(ctx context.Context, lOpts LoggerOpts) context.Context {
	return context.WithValue(ctx, sdkRootTFLoggerOptsKey(), lOpts)
}

// NewSDKSubsystemLoggerWarning is the text included in log output when a
// subsystem is auto-generated by terraform-plugin-log because it was used
// before the SDK instantiated it.
const NewSDKSubsystemLoggerWarning = "This log was generated by an SDK subsystem logger that wasn't created before being used. Use tflog.NewSubsystem to create this logger before it is used."

// GetSDKSubsystemLogger returns the subsystem logger for the named subsystem
// in SDK space. If no such subsystem logger has been created, it will return
// nil.
func GetSDKSubsystemLogger(ctx context.Context, subsystem string) hclog.Logger {
	logger := ctx.Value(sdkSubsystemLoggerKey(subsystem))
	if logger == nil {
		return nil
	}

	hclogger, ok := logger.(hclog.Logger)
	if !ok {
		return nil
	}

	return hclogger
}

// SetSDKSubsystemLogger sets `logger` as the logger for the named subsystem in
// SDK space.
func SetSDKSubsystemLogger(ctx context.Context, subsystem string, logger hclog.Logger) context.Context {
	return context.WithValue(ctx, sdkSubsystemLoggerKey(subsystem), logger)
}

// GetSDKSubsystemTFLoggerOpts retrieves the LoggerOpts of the logger for the named SDK subsystem.
// The value is stored in the context.Context: if none is found, a new one will be created.
func GetSDKSubsystemTFLoggerOpts(ctx context.Context, subsystem string) LoggerOpts {
	lOpts, ok := ctx.Value(sdkSubsystemTFLoggerOptsKey(subsystem)).(LoggerOpts)
	if !ok {
		lOpts = LoggerOpts{}
	}

	return lOpts
}

// SetSDKSubsystemTFLoggerOpts sets the LoggerOpts of the logger for the named SDK subsystem, in the context.
func SetSDKSubsystemTFLoggerOpts(ctx context.Context, subsystem string, lOpts LoggerOpts) context.Context {
	return context.WithValue(ctx, sdkSubsystemTFLoggerOptsKey(subsystem), lOpts)
}
